<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Invoice_pdf_hlc extends CI_Controller {

    public function __construct() {
        parent::__construct();
        $db_group = $this->session->userdata('db_group') ?? 'default';
        $this->db   = $this->load->database($db_group, TRUE);
        $this->load->library('Invoice_order_hlc');
         $this->load->helper('multiyeardb'); 
    }

    /** Sanitize a money/number string like "5,00,000.00" -> 500000.00 */
    private function amt($v) {
        return (float)preg_replace('/[^\d.]/', '', (string)($v ?? '0'));
    }

    /**
     * Decide commission rate (part or split) + RR premium cutoff logic.
     * Cutoff rule: loan_type_id = 5 and (disb_date or batch invoice date) >= 2025-10-02 => premium % = 0
     */
    private function resolveCommissionRateForRow($user_row, $row, $batch_invoice_date) {
        $commissionRate = empty($row['split_partner_id'])
            ? (float)($user_row->part_payout ?? 0)
            : (float)($user_row->split_part_payout ?? 0);

        $invoice_check_date = !empty($user_row->disb_date) ? $user_row->disb_date : $batch_invoice_date;
        if (empty($invoice_check_date)) $invoice_check_date = date('Y-m-d');

        if ((int)$user_row->loan_type_id === 5 && strtotime($invoice_check_date) >= strtotime('2025-10-02')) {
            $commissionRate = 0.0; // cutoff -> premium % becomes zero
        }
        return $commissionRate;
    }

    public function generatePDF($invoice_no) {
        // Fetch batch items
        $data['invoice_items'] = $this->db->select('*')
            ->from('invoices')
            ->where(['inv_batchno' => $invoice_no])
            ->order_by('id', 'asc')
            ->get()
            ->result_array();

        if (empty($data['invoice_items'])) {
            show_error('No invoice items found.');
            return;
        }

        // Header context (from first item)
        $first = reset($data['invoice_items']);
        $connectorID      = $first['connector_id']    ?? null;
        $userID           = $first['user_id']         ?? null;
        $branchID         = $first['branch_id']       ?? null;
        $invoiceDate      = $first['invoice_date']    ?? null;
        $invoiceType      = $first['invoice_type']    ?? null;
        $invoiceNumber    = $first['invoice_number']  ?? null;
        $split_partner_id = $first['split_partner_id']?? null;

        $hlc_user   = $this->db->select("*")->from('mst_user')->where(['id' => $userID])->get()->row();
        $connector  = $this->db->select("*")->from('mst_staff')->where(['staff_id' => ($hlc_user->connector_id ?? null), 'staff_type' => 1])->get()->row();
        $data['branch'] = $this->db->select("*")->from('branch')->where(['id' => $branchID])->get()->row();

        // All payment logs (show all)
        $paid_invoices = $this->db->select('*')
            ->from('invoice_log')
            ->where('invoice_batchno', $invoice_no)
            ->order_by('add_date', 'DESC')
            ->get()
            ->result();

        $split_part_det = null;
        if (!empty($split_partner_id)) {
            $split_part_det = $this->db->select("*")->from('split_partner')->where(['id' => $split_partner_id])->get()->row();
        }

        $data['invoice_date'] = $invoiceDate;
        $data['invoice_type'] = $invoiceType;
        $data['invoice_num']  = $invoiceNumber;

        // TCPDF
        $pdf = new Invoice_order_hlc('P', 'mm', 'A4', true, 'UTF-8', false);
        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetAuthor('Mortgagebuddy');
        $pdf->SetTitle($invoiceNumber);
        $pdf->SetSubject('Invoice');
        $pdf->SetKeywords('Invoice, PDF');

        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
        $pdf->SetFont('dejavusans', '', 6);
        $pdf->SetCellPaddings(1, 2, 1, 1);
        $pdf->SetMargins(10, 20, 10);
        $pdf->AddPage();
        $pdf->SetMargins(10, 45, 10);

        // Header HTML
        if (!empty($split_part_det)) {
            $fromBlock = '
                <strong>To</strong><br>
                Name: ' . htmlspecialchars($split_part_det->staff_fullname ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                Address: <address>' . htmlspecialchars($split_part_det->staff_address ?? '', ENT_QUOTES, 'UTF-8') . '</address><br>
                PAN No.: ' . htmlspecialchars($split_part_det->pan_no ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                GSTIN No.: ' . htmlspecialchars($split_part_det->gst_no ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                Mobile No.: ' . htmlspecialchars($split_part_det->mobile ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                E-mail ID: ' . htmlspecialchars($split_part_det->email ?? '', ENT_QUOTES, 'UTF-8') . '
            ';
        } else {
            $fromBlock = '
                <strong>To</strong><br>
                Name: ' . htmlspecialchars($connector->staff_fullname ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                Address: <address>' . htmlspecialchars($connector->staff_address ?? '', ENT_QUOTES, 'UTF-8') . '</address><br>
                PAN No.: ' . htmlspecialchars($connector->pan_no ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                GSTIN No.: ' . htmlspecialchars($connector->gst_no ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                Mobile No.: ' . htmlspecialchars($connector->mobile ?? '', ENT_QUOTES, 'UTF-8') . '<br>
                E-mail ID: ' . htmlspecialchars($connector->email ?? '', ENT_QUOTES, 'UTF-8') . '
            ';
        }
        
          // Determine if invoice is paid
        $any_paid = false;
        if (!empty($paid_invoices)) {
            foreach ($paid_invoices as $p) {
                if (strtolower(trim($p->status ?? '')) === 'paid') {
                    $any_paid = true;
                    break;
                }
            }
        }

        $html = '
        <table width="100%">
            <tr>
                <td style="text-align:left; padding-left:0;">' . $fromBlock . '</td>
                <td style="text-align:left; padding-left:0;">
                    <strong>From</strong><br>
                    MORTGAGEBUDDY PRIVATE LIMITED<br>
                    H 218, Belapur Station Complex,
                    Navi Mumbai,400614<br>
                    GSTIN NO: 27AAQCM2355K1ZZ<br>
                    HSN Code:
                    
                </td>
                <td style="text-align:left; padding-left:0;">
                    <strong>Date:</strong> ' . (!empty($data['invoice_date']) ? date('F d, Y', strtotime($data['invoice_date'])) : '-') . '<br>
                    <strong>Advice No:</strong> ' . htmlspecialchars($invoiceNumber ?? '-', ENT_QUOTES, 'UTF-8') . '<br>';
                      
                // ✅ Show stamp only if paid
                if ($any_paid) {
                    $html .= '<img src="' . base_url() . 'assets/site_img/paid-stamp.png" height="50" alt="Paid">';
                }
                
                $html .= '
                        </td>
                    </tr>
                </table>';   

        $html .= '
<style>
.table-bordered { border-collapse: collapse; }
.table-bordered th, .table-bordered td { border: 0.1px solid #000; padding: 8px; }
</style>
<br><br><br>
<table class="table table-invoice table-bordered">
    <thead>
        <tr style="background-color:#d9e1f2; text-align:center;">
            <th style="width:4%;">Sr. No</th>
            <th style="width:12%;">Los ID</th>
            <th style="width:13%;">Name Of Customer</th>
            <th>Loan Type</th>
            <th>Branch Name</th>
            <th style="width:11%;">CBS A/C No</th>
            <th>Disburs Date</th>
            <th>Sanctioned Loan Amount</th>
            <th>Commission<br>%</th>
            <th>Commission Amount Rs</th>
        </tr>
    </thead>
    <tbody>
';

        $totalLoanAmt = 0.0;
        $totalCommisionAmt = 0.0;

        foreach ($data['invoice_items'] as $key => $row) {
            $user_data = $this->db->select("*")->from('mst_user')->where(['id' => $row['user_id']])->get()->row();
            if (!$user_data) continue;

            $type_data = $this->db->select("*")->from('mst_loan_type')->where(['id' => $user_data->loan_type_id])->get()->row();
            $branch    = $this->db->select("*")->from('branch')->where(['id' => $user_data->branch_id])->get()->row();
            $split_user_invoice = $this->db->select("*")->from('invoices')->where(['id' => $user_data->invoice_id])->get()->row();
            $split_main_invoice = $this->db->select("*")->from('invoices')->where(['id' => $user_data->split_invoice_id])->get()->row();

            $loanTypeId = (int)$user_data->loan_type_id;
            $sanctionAmt = $this->amt($user_data->sanction_amt);
            $rinnAmt     = $this->amt($user_data->rinnraksha_amt);

            // Accumulate total loan base
            $totalLoanAmt += ($loanTypeId === 5) ? $rinnAmt : $sanctionAmt;

            // Commission rate resolution (handles RR cutoff)
            $commissionRate = $this->resolveCommissionRateForRow($user_data, $row, $invoiceDate);

            // ---- Split the amounts explicitly so subtotal matches what’s shown ----
            // 1) Premium % for RR (eligible only)
            $premiumPercentAmt = 0.00;
            if ($loanTypeId === 5 && strcasecmp($user_data->rinnraksha_status ?? '', 'Yes') === 0 && empty($user_data->applicant_rr_id)) {
                $premiumPercentAmt = round($rinnAmt * (float)$commissionRate / 100, 2); // cutoff can make it zero
            }

            // 2) Normal % / flat for non-RR rows
            $normalPercentOrFlat = 0.00;
            if ($loanTypeId !== 5) {
                if ($loanTypeId === 4 && $sanctionAmt <= 1000000) {
                    $normalPercentOrFlat = 2000.00;
                } else {
                    $normalPercentOrFlat = round($sanctionAmt * (float)$commissionRate / 100, 2);
                }
            }

            // 3) RR fixed (slab via helper) for RR eligible rows
            $rrFixed = 0.00;
            if ($loanTypeId === 5 && strcasecmp($user_data->rinnraksha_status ?? '', 'Yes') === 0 && empty($user_data->applicant_rr_id)) {
                $financial_year = $this->session->userdata('financial_year');
                $rrFixed = (float)calculate_rr_commission_multi_year($financial_year, $user_data);
            }

            // main row display amount (default)
            $mainDisplayAmt = ($loanTypeId === 5) ? $premiumPercentAmt : $normalPercentOrFlat;

            // true row commission -> subtotal/tax/tds base
            $rowCommission = round($premiumPercentAmt + $normalPercentOrFlat + $rrFixed, 2);
            $totalCommisionAmt += $rowCommission;

            // show %; for flat top-up show '-'
            $showPercent = ($loanTypeId === 4 && $sanctionAmt <= 1000000)
                ? '-'
                : number_format((float)$commissionRate, 2, '.', '') . '%';

            // ---------- RR Fixed display rules ----------
            $invoice_check_date = !empty($user_data->disb_date) ? $user_data->disb_date : $invoiceDate;
            if (empty($invoice_check_date)) { $invoice_check_date = date('Y-m-d'); }

            // AFTER cutoff: merge RR fixed into last cell (02-10-2025 excluded)
            $showRRFixedAfterCutoff = (
                $loanTypeId === 5 &&
                $rrFixed > 0 &&
                strtotime($invoice_check_date) > strtotime('2025-10-02')
            );

            // BEFORE cutoff: show legacy EXTRA ROW (strictly before 02-10-2025)
            $showLegacyRRRowBeforeCutoff = (
                $loanTypeId === 5 &&
                $rrFixed > 0 &&
                strtotime($invoice_check_date) < strtotime('2025-10-02')
            );

            // Amount & label for the MAIN row
            $finalAmount  = $showRRFixedAfterCutoff ? $rrFixed : $mainDisplayAmt;
            $finalPercent = $showRRFixedAfterCutoff ? 'Fixed' : $showPercent;

            // Main row HTML
            $html .= '
            <tr>
                <td style="width:4%;">' . ($key + 1) . '</td>
                <td style="width:12%;">' . htmlspecialchars($user_data->los_id ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                <td style="width:13%;">' . htmlspecialchars($user_data->user_name ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                <td>' . htmlspecialchars($type_data->name ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                <td>' . htmlspecialchars(($branch->branch_name ?? '') . ' [' . ($branch->branch_code ?? '') . ']', ENT_QUOTES, 'UTF-8') . '</td>
                <td style="width:11%;">' . (($loanTypeId === 5) ? htmlspecialchars($user_data->rr_cbs_acct_no ?? '', ENT_QUOTES, 'UTF-8') : htmlspecialchars($user_data->bank_acct_no ?? '', ENT_QUOTES, 'UTF-8')) . '</td>
                <td>' . (!empty($user_data->disb_date) ? date('d.m.Y', strtotime($user_data->disb_date)) : '-') . '</td>
                <td>' . formatIndianNumber(($loanTypeId === 5) ? $rinnAmt : $sanctionAmt) . '</td>
                <td>' . $finalPercent . '</td>
                <td>' . formatIndianNumber($finalAmount) . '</td>
            </tr>';

            // Legacy EXTRA RR row only BEFORE cutoff
            if ($showLegacyRRRowBeforeCutoff) {
                $html .= '
                <tr>
                    <td></td>
                    <td>' . htmlspecialchars($user_data->los_id ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                    <td>' . htmlspecialchars($user_data->user_name ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                    <td style="text-align:left;">RR Commission (Fixed)</td>
                    <td></td><td></td><td></td><td></td><td></td>
                    <td>' . formatIndianNumber($rrFixed) . '</td>
                </tr>';
            }

            // Split partner invoice note (optional)
            if (!empty($row['split_partner_id']) && !empty($split_user_invoice->invoice_number)) {
                $html .= '<tr><td colspan="10" style="text-align:left;"><em>Split Partner Invoice No: ' . htmlspecialchars($split_user_invoice->invoice_number, ENT_QUOTES, 'UTF-8') . '</em></td></tr>';
            } elseif (!empty($user_data->split_invoice_id) && !empty($split_main_invoice->invoice_number)) {
                $html .= '<tr><td colspan="10" style="text-align:left;"><em>Split Partner Invoice No: ' . htmlspecialchars($split_main_invoice->invoice_number, ENT_QUOTES, 'UTF-8') . '</em></td></tr>';
            }
        } // foreach rows

        // -------- Taxes (GST) --------
        // Determine GST owner: split partner (batch-level) else connector
        $gst_owner = !empty($split_part_det) ? $split_part_det : $connector;
        $owner_gst_no = isset($gst_owner->gst_no) ? trim($gst_owner->gst_no) : '';
        $owner_gst_apply_date = isset($gst_owner->gst_apply_date) ? trim($gst_owner->gst_apply_date) : '';
        if ($owner_gst_apply_date === '0000-00-00') $owner_gst_apply_date = '';

        $apply_gst = false;
        if ($owner_gst_no !== '') {
            if ($owner_gst_apply_date === '') {
                $apply_gst = true;
            } else {
                try {
                    $invDate  = new DateTime($invoiceDate);
                    $gstApply = new DateTime($owner_gst_apply_date);
                    $apply_gst = ($invDate >= $gstApply);
                } catch (Exception $e) {
                    $apply_gst = false; // conservative
                }
            }
        }

        $mb_state_code = 27; // MH
        $owner_state_id = isset($gst_owner->state_id) ? (int)$gst_owner->state_id : null;

        $cgst = $sgst = $igst = 0.00;
        if ($apply_gst && $totalCommisionAmt > 0) {
            if (!is_null($owner_state_id) && $owner_state_id === $mb_state_code) {
                $cgst = round(0.09 * $totalCommisionAmt, 2);
                $sgst = round(0.09 * $totalCommisionAmt, 2);
            } else {
                $igst = round(0.18 * $totalCommisionAmt, 2);
            }
        }
        $gst = round($cgst + $sgst + $igst, 2);

        // -------- TDS --------
        $netPayamt = $totalCommisionAmt;
        $invoiceTimestamp = !empty($invoiceDate) ? strtotime($invoiceDate) : time();
        $thresholdDate = strtotime('2024-10-01');

        if ($invoiceTimestamp >= $thresholdDate) {
            $tdsPer = 2;
            $tds = round($netPayamt * 0.02, 2);
        } else {
            $tdsPer = 5;
            $tds = round($netPayamt * 0.05, 2);
        }

        $grossTotal = round($netPayamt + $gst, 2);
        $netTotal   = round($grossTotal - $tds, 2);

        // Totals rows
        $html .='
        <tr><td colspan="10" style="height:8px;"></td></tr>
        <tr>
           <td colspan="8"></td>
           <td>-</td>
           <td>-</td>
        </tr>
        <tr>
            <td colspan="5" rowspan="7">';

            if (!empty($paid_invoices)) {
                // check if any entry is fully paid to show stamp
                $any_paid = false;

                $html .= '
                <table style="width:100%; border-collapse:collapse;" border="0" cellpadding="6">
                    <thead>
                        <tr>
                            <th style="text-align:left;">Status</th>
                            <th style="text-align:left;">Payment Date</th>
                            <th style="text-align:right;">Amount (Rs)</th>
                            <th style="text-align:left;">Ref No / Cheq No</th>
                        </tr>
                    </thead>
                    <tbody>';

                foreach ($paid_invoices as $p) {
                    $any_paid = $any_paid || (strtolower(trim($p->status ?? '')) === 'paid');

                    $html .= '<tr>
                        <td>'.htmlspecialchars(ucwords($p->status ?? 'Unpaid'), ENT_QUOTES, 'UTF-8').'</td>
                        <td>'.(!empty($p->payment_date) ? date('d-m-Y', strtotime($p->payment_date)) : '-').'</td>
                        <td style="text-align:right;">'.formatIndianNumber($this->amt($p->pay_amt ?? 0)).'</td>
                        <td>'.htmlspecialchars($p->bank_details ?? '', ENT_QUOTES, 'UTF-8').'</td>
                    </tr>';
                }

                $html .= '</tbody></table>';

                // Optional: "PAID" stamp agar koi entry paid ho
               /* if ($any_paid) {
                    $html .= '<div style="margin-top:8px;">
                        <img src="'.base_url().'assets/site_img/paid-stamp.png" height="100" alt="Paid">
                   </div>';
                }*/
            } else {
                $html .= 'unpaid';
            }

            $html .= '</td>
            <td colspan="2" style="text-align:right;"><b>Sub Total</b></td>
            <td><strong>'.formatIndianNumber($totalLoanAmt).'</strong></td>
            <td></td>
            <td><strong>'.formatIndianNumber($totalCommisionAmt).'</strong></td>
            </tr>

        <tr>
           <td colspan="2" style="text-align:right;"><b>Tax Amount - CGST 9%</b></td>
           <td></td>
           <td>-</td>
           <td>'. (empty($cgst) ? '-' : formatIndianNumber($cgst)) . '</td>
        </tr>
        <tr>
           <td colspan="2" style="text-align:right;"><b>Tax Amount - SGST 9%</b></td>
           <td></td>
           <td>-</td>
           <td>'. (empty($sgst) ? '-' : formatIndianNumber($sgst)) . '</td>
        </tr>
         <tr>
           <td colspan="2" style="text-align:right;"><b>Tax Amount - IGST 18%</b></td>
           <td></td>
           <td>-</td>
           <td>'. (empty($igst) ? '-' : formatIndianNumber($igst)) . '</td>
        </tr>
        <tr>
           <td colspan="2" style="text-align:right;"><b>Gross Total</b></td>
           <td></td>
           <td></td>
           <td><strong>'. formatIndianNumber($grossTotal).'</strong></td>
        </tr>
        <tr>
           <td colspan="2" style="text-align:right;"><b>Less: TDS @ '.$tdsPer.'%</b></td>
           <td></td>
           <td></td>
           <td><strong>'.formatIndianNumber($tds).'</strong></td>
        </tr>
        <tr>
           <td colspan="2" style="text-align:right;"><b>Net Total</b></td>
           <td></td>
           <td></td>
           <td><strong>'.formatIndianNumber($netTotal).'</strong></td>
        </tr>
    </tbody>
</table>';

        // Footer note + bank details
        $html .='<p>I confirm that the above housing proposal has been disbursed from the mentioned branch on MortgageBuddy Private Limited HLC code (MUMHLC03147).</p>
        <br><br>
        <table class="table">
            <tr class="align-items-center mt-5">
                <td width="40%">
                    <p class="mb-0"><strong>Rs.(In words): '. ucfirst(getIndianCurrency($netTotal)) . ' only</strong></p>
                </td>
                <td width="30%"></td>
                <td width="40%" style="text-align:right;">
                    <p class="mb-0">Signature</p>
                </td>
            </tr>
        </table>
        <br><br><br><br>';

        // Bank block (split partner takes precedence)
        $bank = !empty($split_part_det) ? $split_part_det : $connector;
        $html .= '<table style="width:30%;">
            <tr><td><strong>Type of Account :</strong></td><td>'.htmlspecialchars($bank->bank_acct_type ?? '', ENT_QUOTES, 'UTF-8').'</td></tr>
            <tr><td><strong>Account No. :</strong></td><td>'.htmlspecialchars($bank->bank_acct_no ?? '', ENT_QUOTES, 'UTF-8').'</td></tr>
            <tr><td><strong>IFSC code:</strong></td><td>'.htmlspecialchars($bank->bank_ifsc ?? '', ENT_QUOTES, 'UTF-8').'</td></tr>
        </table>';

        // Output
        $pdf->writeHTML($html, true, false, true, false, '');
        $pdf->Output(($invoiceNumber ?? 'invoice').'.pdf', 'I');
    }

    // (Optional) keep your OLD method if you want to compare outputs
    // public function generatePDF_OLD_DEE($invoice_no) { ... }
}
